function [CvM,AD,CvML,ADL] = goodnessfit(x,cparm,nfft)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code: Used for computing results in Table 4.2 
%              (see also Exercise 4.4b)
% File: goodnessfit.m
%
% Goodness-of-fit tests statistics CvM and AD for testing 
% Gaussianity and Linearity based on the paper by
% Jahan, N. and Harvill, J.L. (2008).
%
% INPUT:
% x     = input time series: column vector
% cparm = resolution (smoothing) parameter c; 
%         0.5<c<1.0 [default=0.51], increasing c reduces the 
%         variance, but increases the bias.
% nfft  = FFT length [default = 128]
%
% OUTPUT:
% CvM and AD test statistics for the Gaussianity hypothesis
% CvML and ADL test statistics for the linearity hypothesis
%
% References:
% Hinich, M.J. (1982). Testing for Gaussianity and linearity of 
%   stationary time series. 
%   Journal of Time Series Analysis, 3(3), 169-176.
%   DOI: 10.1111/j.1467-9892.1982.tb00339.x.
% Jahan, N. and Harvill, J.L. (2008). Bispectral-based  
%   goodness-of-fit tests of Gaussianity and linearity of 
%   stationary time series. Communications in Statistics: 
%   Theory and Methods, 37(20), 3216-3227.
%   DOI: 10.1080/03610920802133319.
% Stephens, M.A. (1986). Tests based on EDF statistics. 
%   In: D'Agostino, R.B. and Stephens, M.A. (Eds.): 
%   Goodness-of-Fit Techniques. Marcel Dekker, New York. 
% ---------------- parameter checks: -----------------------------
    if (min(size(x)) ~= 1)
       error('x: should be a vector')
    end
    nsamp = length(x);
    if (exist('nfft') ~= 1) 
       nfft = 128;
    end
    
    nrecs = floor(nsamp/nfft);
    nrecs = max(nrecs,1);
    ksamp = min(nfft,nsamp);
    if (nfft  > nsamp)
      disp(' glstat results are unreliable if zero-padding is severe: ')
      disp(' fft length= ',int2str(nfft),' data length=',int2str(nsamp))
    end

    if (exist ('cparm') ~= 1) 
      cparm = 0.51; 
    end
    if (cparm >= 1.0)
       error('cparm  cannot be greater than or equal to 1.')
    end
    if (cparm <= 0.5 && nrecs == 1)
       error('cparm: for single segments: allowed range is (0.5,1.0)')
    end

    M = fix(nfft^cparm);  M = M + 1 - rem(M,2);
    Msmuth = M;
% ----------------- Estimate raw bispectrum ------------------------
    mrow = fix(nfft/3)+1; ncol = fix(nfft/2)+1;
    F    = zeros(mrow,ncol);
    S    = zeros(nfft,1);

    mask = hankel(1:mrow,mrow:mrow+ncol-1);
    for k=1:nrecs
        y  = x(1:ksamp);
        xf = fft(y-mean(y), nfft);
        xc = conj(xf);
        S  = S + xf .* xc;                    % Power spectrum
        F  = F + xf(1:mrow) * xf(1:ncol).' .* ...
             reshape (xc(mask), mrow, ncol) ;
    end
    F = F/(nfft*nrecs);   % `Raw' bispectrum
    S = S/(nfft*nrecs);   % `Raw' power spectrum

% ---------- Zero out area outside principal domain ---------------
    F(1:mrow,1:mrow) = triu(F(1:mrow,1:mrow));
    Q = ones(mrow, ncol);
    Q(1:mrow,1:mrow) = triu(Q(1:mrow,1:mrow)) + eye(mrow);

%   The 2f1 + f2 = 1 line:
    r = (rem(nfft,3) == 2);        
    for k=mrow+1:ncol-r
        j = k-mrow;
        Q(mrow-2*j+2:mrow, k) = zeros(2*j-1,1);
        F(:, k) = F(:,k).* Q(:,k);
        Q(mrow-2*j+1, k) = 2;   % Factor 2 on boundary
    end

    F = F(2:mrow,2:ncol);       % In principal domain, no dc terms
    Q = Q(2:mrow,2:ncol);
    mrow = mrow-1;
    ncol = ncol-1;
% -------- Smooth the estimated spectrum and bispectrum ------------
%   Msmuth * Msmuth box-car smoother
%   only every Msmuth term in the smoothed output is retained
%   "independent" estimates

    M  = Msmuth;
    m1 = rem(mrow,M); m2=rem(ncol,M);
    m1 =  - m1 + M * (m1 ~= 0);
    m2 =  - m2 + M * (m2 ~= 0);
    F  = [F,zeros(mrow,m2);zeros(m1,ncol+m2)];
    Q  = [Q,zeros(mrow,m2);zeros(m1,ncol+m2)];
    k  = size(F)/Msmuth;
    k1 = k(1); k2 = k(2);

% Apply the boxcar smoother
% can replace the next ten lines or so with
% B = kron(eye(k1),ones(1,Msmuth))*F*kron(eye(k2),ones(Msmuth,1))
%     /Msmuth^2;
% Q = kron(eye(k1),ones(1,Msmuth))*Q*kron(eye(k2),ones(Msmuth,1));

    ind = 1:Msmuth;
    B   = zeros(k1,k2); Q1 = B;
    for i=1:k1
       for j=1:k2
          t       = F((i-1)*Msmuth+ind,(j-1)*Msmuth+ind );
          B(i,j)  = mean(t(:));
          t       = Q((i-1)*Msmuth+ind,(j-1)*Msmuth+ind );
          Q1(i,j) = sum(t(:));
      end
    end
    Q = Q1;
%  --------------
%   At this point B corresponds to B in eqn.(2.6) of Hinich's 
%   paper and Q corresponds to the definition following eqn. (2.8)
%  --------------
    M    = Msmuth;
    S    = conv(S, ones(M,1))/M;
    S    = S(M+1:M:M+nfft-1);
    S1   = S(1:k1) * S(1:k2)'.* hankel(S(2:k1+1),S(k1+1:k1+k2)); 
    S1   = ones(k1,k2) ./ S1;
    ind  = find(Q > 0);
    Q    = Q(ind);
    Bic = S1(ind) .* abs(B(ind)).^2; % Squared bicoherence
    scale = 2 * (Msmuth^4) / nfft;
    Xstat = scale * Bic ./ Q;        % 2|X_{m,n}|^2, eqn.(2.9)
% -----------------------------------------------------------------
% ---------------------- Gaussianity EDF tests with fully     -----
%                        specified distribution CvM and AD
% -----------------------------------------------------------------
    Xsort   = sort(Xstat);
    q       = 0.5*erfc(-Xsort/sqrt(2)); % Normal CDF
    Mlength = length(Xstat);            % Length M

    ind = 0;                 
    for i = 1:Mlength
        if(q(i)<1 && q(i)>0)       % Remove zeros and ones
          ind       = ind+1;
          qnew(ind) = q(i);
        end
    end

    mAD = length(qnew);  % Effective length AD test (Not used)
    m   = Mlength;       % Length CvM test
   
    s1 = 0;
    for i = 1:m
        s1 = s1+(q(i)-(2*i-1)/2/m)^2;
    end
    CvM1 = (s1+1/12/m)/m;
    
%   Modification for a fully specified distribution    
    CvM  = (CvM1-0.4/m+0.6/m^2)*(1+1/m);   

    if(CvM >= 0.00 && CvM < 0.0275);
       pvCvMG = 1- exp(-13.953 + 775.5*CvM - 12542.61*CvM^2);
    elseif(CvM >= 0.0275 && CvM < 0.051);
       pvCvMG = 1- exp(-5.903 +179.546*CvM -1515.29*CvM^2);
    elseif(CvM < 0.092 && CvM > 0.051);
       pvCvMG = exp(0.886 - 31.62*CvM + 10.897*CvM^2);  
    elseif(CvM > 0.092);
       pvCvMG = exp(1.111 - 34.242*CvM + 12.832*CvM^2);
    end

    fx = normcdf(Xsort,mean(Xsort),std(Xsort));
    i  = 1:m;   
    S  = sum((((2*i)-1)/m)*(log(fx)+log(1-fx(m+1-i))));
    AD = -m-S;
    
%   Correction factor for small sample sizes: case normal    
    ADa = AD*(1 + 0.75/m + 2.25/m^2); 
    if (ADa >= 0.00 && ADa < 0.200);
        pvADG = 1 - exp(-13.436 + 101.14*ADa - 223.73*ADa^2);
    elseif (ADa >= 0.200 && ADa < 0.340);
        pvADG = 1 - exp(-8.318 + 42.796*ADa - 59.938*ADa^2);
    elseif (ADa >= 0.340 && ADa < 0.600);
        pvADG = exp(0.9177 - 4.279*ADa - 1.38*ADa^2);
    else (ADa >= 0.600 && ADa <= 13);
        pvADG = exp(1.2937 - 5.709*ADa + 0.0186*ADa^2);
    end
    
disp(['CvM Gaussianity test= ', num2str(CvM),', ...
p-value CvM =',num2str(pvCvMG),', AD Gaussianity test= ',...
num2str(AD), ', p-value AD = ',num2str(pvADG)])  
% ----------------------------------------------------------------
% --------------------- Linearity EDF tests CvM and AD  ----------
% ----------------------------------------------------------------
    df    = 2;
    xmean = mean(Xstat); 
    if(xmean>df)
      lam = xmean-df;
    else
      lam = 0;
    end  
    h = 1-(2*(df+lam)*(df+3*lam)/(3*(df+2*lam)^2));  % Exponent  
    Ytrans1 = (Xstat/(df+lam));         % Transformed variables
    for i = 1:Mlength 
        Ytrans(i) = (Ytrans1(i)^(h));
    end
  
    muY1 = 1+h*(h-1)*(df+2*lam)/(df+lam)^2;
    muY  = muY1-h*(h-1)*(2-h)*(1-3*h)*((df+2*lam)^2)/(2*(df+lam)^4); 
  
    sigma2Y1 = h*h*(2*(df+2*lam))/((df+lam)^2);
    sigma2Y  = sigma2Y1*(1-(1-h)*(1-3*h)*(df+2*lam)/((df+lam)^2));   
  
    for i = 1:Mlength
        Ystand(i) = (Ytrans(i)-muY)/sqrt(sigma2Y);     
    end
    Ysort = sort(Ystand);

    qlin = 0.5*erfc(-Ysort/sqrt(2));  % Normal CDF
    mlin = length(Ysort);             % Effective length M  
  
    i    = (1:mlin);
    CvML = (1/12/mlin+sum((qlin-(2*i-1)/2/mlin).^2))/mlin;    

%   Modification of CvM for unknown mean and variance 
%   as given in Table 4.7 of Stephens (1986)
    CvML = CvML*(1+0.5/mlin);  
    
%   Asymptotic p-values for modified CvM as given in Table 4.9
%   of Stephens (1986)  
    if(CvML >= 0.00 && CvML < 0.0275);
       pvCvML = 1- exp(-13.953 + 775.5*CvML - 12542.61*CvML^2);
    elseif(CvML >= 0.0275 && CvML < 0.051);
       pvCvML = 1- exp(-5.903 +179.546*CvML -1515.29*CvML^2);
    elseif(CvML < 0.092 && CvML >0.051);
       pvCvML = exp(0.886 - 31.62*CvML + 10.897*CvML^2);  
    elseif(CvML > 0.092);
       pvCvML = exp(1.111 - 34.242*CvML + 12.832*CvML^2);
    end
   
    fxs = normcdf(Ysort,mean(Ysort),std(Ysort))';
    i   = (1:mlin);    
    S   = sum((((2*i)-1)/mlin)*(log(fxs)+log(1-fxs(mlin+1-i))));
    ADL = -mlin-S; 

%   Modification of AD test for unknown mean and variance               
    ADL = ADL*(1+0.75/mlin+2.25/(mlin^2)); 
    if(ADL >= 0.00 && ADL < 0.200);
        pvADL =  1 - exp(-13.436 + 101.14*ADL - 223.73*ADL^2);
    elseif (ADL >= 0.200 && ADL < 0.340);
        pvADL = 1 - exp(-8.318 + 42.796*ADL - 59.938*ADL^2);            
    elseif(ADL < 0.600 && ADL >= 0.340)
        pvADL = exp(0.9177 - 4.279*ADL - 1.38*ADL^2); 
    elseif(ADL >= 0.600 && ADL<= 13)
        pvADL=exp(1.2937 - 5.709*ADL + 0.0186*ADL^2);
    end

disp(['CvM Linearity test =  ',num2str(CvML),', p-value CvM = ',
...,num2str(pvCvML),', AD Linearity test = ', num2str(ADL), ',
..., p-value AD = ', num2str(pvADL)])